<?php

/**
 * DokuWiki Plugin struct (Action Component)
 *
 * @license GPL 2 http://www.gnu.org/licenses/gpl-2.0.html
 * @author  Andreas Gohr, Michael Große <dokuwiki@cosmocode.de>
 */

use dokuwiki\Extension\ActionPlugin;
use dokuwiki\Extension\EventHandler;
use dokuwiki\Extension\Event;
use dokuwiki\plugin\struct\meta\AccessTable;
use dokuwiki\plugin\struct\meta\Assignments;
use dokuwiki\plugin\struct\meta\Schema;
use dokuwiki\plugin\struct\meta\Search;
use dokuwiki\plugin\struct\types\Lookup;

/**
 * Handles bureaucracy additions
 *
 * This registers to the template action of the bureaucracy plugin and saves all struct data
 * submitted through the bureaucracy form to all newly created pages (if the schema applies).
 *
 * It also registers the struct_schema type for bureaucracy which will add all fields of the
 * schema to the form. The struct_field type is added through standard naming convention - see
 * helper/fiels.php for that.
 */
class action_plugin_struct_bureaucracy extends ActionPlugin
{
    /**
     * Registers a callback function for a given event
     *
     * @param EventHandler $controller DokuWiki's event controller object
     * @return void
     */
    public function register(EventHandler $controller)
    {
        $controller->register_hook('PLUGIN_BUREAUCRACY_PAGENAME', 'BEFORE', $this, 'handleLookupFields');
        $controller->register_hook('PLUGIN_BUREAUCRACY_EMAIL_SEND', 'BEFORE', $this, 'handleLookupFields');
        $controller->register_hook('PLUGIN_BUREAUCRACY_TEMPLATE_SAVE', 'AFTER', $this, 'handleSave');
        $controller->register_hook('PLUGIN_BUREAUCRACY_FIELD_UNKNOWN', 'BEFORE', $this, 'handleSchema');
    }

    /**
     * Load a whole schema as fields
     *
     * @param Event $event event object by reference
     * @param mixed $param [the parameters passed as fifth argument to register_hook() when this
     *                           handler was registered]
     * @return bool
     */
    public function handleSchema(Event $event, $param)
    {
        $args = $event->data['args'];
        if ($args[0] != 'struct_schema') return false;
        $event->preventDefault();
        $event->stopPropagation();

        /** @var helper_plugin_bureaucracy_field $helper */
        $helper = plugin_load('helper', 'bureaucracy_field');
        $helper->initialize($args);

        $schema = new Schema($helper->opt['label']);
        if (!$schema->getId()) {
            msg('This schema does not exist', -1);
            return false;
        }

        foreach ($schema->getColumns(false) as $column) {
            /** @var helper_plugin_struct_field $field */
            $field = plugin_load('helper', 'struct_field');
            // we don't initialize the field but set the appropriate values
            $field->opt = $helper->opt; // copy all the settings to each field
            $field->opt['label'] = $column->getFullQualifiedLabel();
            $field->column = $column;
            $event->data['fields'][] = $field;
        }
        return true;
    }

    /**
     * Replace lookup fields placeholder's values
     *
     * @param Event $event event object by reference
     * @param mixed $param [the parameters passed as fifth argument to register_hook() when this
     *                           handler was registered]
     * @return bool
     */
    public function handleLookupFields(Event $event, $param)
    {
        foreach ($event->data['fields'] as $field) {
            if (!is_a($field, 'helper_plugin_struct_field')) continue;
            if (!$field->column->getType() instanceof Lookup) continue;

            $value = $field->getParam('value');
            if (!is_array($value)) $value = [$value];

            $config = $field->column->getType()->getConfig();

            // find proper value
            // current Search implementation doesn't allow doing it using SQL
            $search = new Search();
            $search->addSchema($config['schema']);
            $search->addColumn($config['field']);
            $result = $search->getRows();
            $pids = $search->getPids();
            $rids = $search->getRids();

            $field->opt['struct_pids'] = [];
            $new_value = [];
            foreach ($value as $pid) {
                $counter = count($result);
                for ($i = 0; $i < $counter; $i++) {
                    // lookups can reference pages or global data, so check both pid and rid
                    // make sure not to double decode pid!
                    $originalPid = $pid;
                    // do not throw JSON exception here, we supply alternative values if json_decode doesn't
                    $pid = json_decode($pid, null, 512)[0] ?? $pid;
                    $rid = json_decode($originalPid, null, 512)[1] ?? null;
                    if (($pid && $pids[$i] === $pid) || ($rid && $rids[$i] === $rid)) {
                        $field->opt['struct_pids'][] = $pid;
                        $new_value[] = $result[$i][0]->getDisplayValue();
                    }
                }
            }

            //replace previous value
            if ($field->column->isMulti()) {
                $field->opt['value'] = $new_value;
            } else {
                $event->data['values'][$field->column->getFullQualifiedLabel()] = $new_value[0] ?? '';
            }
        }
        return true;
    }

    /**
     * Save the struct data
     *
     * @param Event $event event object by reference
     * @param mixed $param [the parameters passed as fifth argument to register_hook() when this
     *                           handler was registered]
     * @return bool
     */
    public function handleSave(Event $event, $param)
    {
        // get all struct values and their associated schemas
        $tosave = [];
        foreach ($event->data['fields'] as $field) {
            if (!is_a($field, 'helper_plugin_struct_field')) continue;
            /** @var helper_plugin_struct_field $field */
            $tbl = $field->column->getTable();
            $lbl = $field->column->getLabel();
            if (!isset($tosave[$tbl])) $tosave[$tbl] = [];

            if ($field->column->isMulti() && $field->column->getType() instanceof Lookup) {
                $tosave[$tbl][$lbl] = $field->opt['struct_pids'];
            } else {
                $tosave[$tbl][$lbl] = $field->getParam('value');
            }
        }

        // save all the struct data of assigned schemas
        $id = $event->data['id'];
        $time = filemtime(wikiFN($id));

        $assignments = Assignments::getInstance();
        $assigned = $assignments->getPageAssignments($id);
        foreach ($tosave as $table => $data) {
            if (!in_array($table, $assigned)) continue;
            $access = AccessTable::getPageAccess($table, $id, $time);
            $validator = $access->getValidator($data);
            if ($validator->validate()) {
                $validator->saveData($time);

                // make sure this schema is assigned
                $assignments->assignPageSchema(
                    $id,
                    $validator->getAccessTable()->getSchema()->getTable()
                );

                // trigger meta data rendering to set page title
                p_get_metadata($id);
            }
        }

        return true;
    }
}

// vim:ts=4:sw=4:et:
